package Game::HeroesVsAliens;

use 5.006;
use strict;
use warnings;
our $VERSION = '0.02';

use Moo;

use Game::HeroesVsAliens::Mouse;
use Game::HeroesVsAliens::Cell;
use Game::HeroesVsAliens::Hero;
use Game::HeroesVsAliens::Alien;
use Game::HeroesVsAliens::Message;
use Game::HeroesVsAliens::Bonus;

use SDL::Event;
use SDLx::App;
use SDL::GFX::Primitives;
use SDLx::Text;
use SDL::GFX::Primitives;
use Data::LnArray;

with 'Game::HeroesVsAliens::Role::ResourceDirectory';

has game_over => (
	is => 'rw',
	default => sub { 0 }
);

has game_grid => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has heroes => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has aliens => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has messages => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has bonuses => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has alien_positions => (
	is => 'rw',
	default => sub { Data::LnArray->new; }
);

has aliens_interval => (
	is => 'rw',
	default => sub { 400 }
);

has frame => (
	is => 'rw',
	default => sub { 0 }
);

has cell_size => (
	is => 'rw',
	default => sub { 100 }
);

has cell_gap => (
	is => 'rw',
	default => sub { 3 }
);

has app_width => (
	is => 'rw',
	default => sub { 800 }
);

has app_height => (
	is => 'rw',
	default => sub { 600 }
);

has score => (
	is => 'rw',
	default => sub { 0 }
);

has winning_score => (
	is => 'rw',
	default => sub { 500 }
);

has score_text => (
	is => 'rw',
	default => sub {
		my $score = SDLx::Text->new(
			size => 18,
			x => 10,
			y => 10
		);
		return $score;
	}
);

has resources => (
	is => 'rw',
	default => sub { 1000 }
);

has resource_text => (
	is => 'rw',
	default => sub {
		my $resource = SDLx::Text->new(
			size => 18,
			x => 10,
			y => 30
		);
		return $resource;
	}
);

has ctx => (
	is => 'rw'
);

has background => (
	is => 'rw',
	default => sub {
		my $sprite = SDLx::Sprite->new(
			image => $_[0]->resource_directory . "/" . 'resources/game-background.png',
		);
		$sprite->rect->x(0);
		$sprite->rect->y(0);
		return $sprite;
	}
);

has mouse => (
	is => 'rw',
	default => sub {
		return Game::HeroesVsAliens::Mouse->new();
	}
);

sub run {
	my ($self) = @_;

	$self->ctx(
		SDLx::App->new(
			title => 'Heroes Vs Aliens',
			width => $self->app_width,
			height => $self->app_height,
		)
	);

	$self->resource_directory;

	$self->ctx->add_event_handler(sub {
		my $e = shift;
		if ($e->type == SDL_MOUSEMOTION) {
			$self->mouse->x($e->motion_x);
			$self->mouse->y($e->motion_y);
		}
		if ($e->type == SDL_MOUSEBUTTONDOWN) {
			$self->add_hero();
		}
		$self->ctx->stop if ($e->type == SDL_QUIT);
	});

	$self->ctx->add_show_handler(sub {
		my $e = shift;
		if (!$self->game_over) {
			$self->background->draw($self->ctx);
			$self->show_grid();
			$self->handle_aliens();
			$self->handle_heroes();
			$self->handle_messages();
			$self->handle_bonuses();
			$self->handle_game_status();
			$self->frame($self->frame + 1);
			$self->ctx->update;
		}
	});

	$self->create_grid();

	$self->ctx->run();
}


sub create_grid {
	my ($self) = @_;
	for (my $y = $self->cell_size; $y < $self->app_height; $y += $self->cell_size) {
		for (my $x = 0; $x < $self->app_width; $x += $self->cell_size) {
			$self->game_grid->push(Game::HeroesVsAliens::Cell->new({
				x => $x,
				y => $y
			}));
		}
	}
}

sub show_grid {
	my ($self) = @_;
	for (@{$self->game_grid}) {
		$_->draw($self);
	}
}

sub handle_game_status {
	my $self = shift;
	$self->score_text->write_to($self->ctx, "Score: " . $self->score);
	$self->resource_text->write_to($self->ctx, "Resources: " . $self->resources);
	if ($self->game_over) {
		$self->draw_text(
			130,
			200,
			100,
			[255, 255, 255, 255],
			'Game Over'
		);
	}
	if ($self->score >= $self->winning_score && $self->aliens->length == 0) {
		$self->draw_text(
			130,
			200,
			100,
			[255, 255, 255, 255],
			'Level Complete'
		);
		$self->draw_text(
			130,
			300,
			50,
			[255, 255, 255, 255],
			'You win with ' . $self->score . ' points'
		);
		$self->game_over(1);
	}
}

sub add_hero {
	my $self = shift;
	my $pos_x = $self->mouse->x - ($self->mouse->x % $self->cell_size) + $self->cell_gap;
	my $pos_y = $self->mouse->y - ($self->mouse->y % $self->cell_size) + $self->cell_gap;
	return if ($pos_y < $self->cell_size);
	for my $hero (@{$self->heroes}) {
		if ($hero->x == $pos_x && $hero->y == $pos_y) {
			return;
		}
	}

	my $hero_cost = 100;
	if ($self->resources >= $hero_cost) {
		$self->heroes->push(Game::HeroesVsAliens::Hero->new(
			x => $pos_x,
			y => $pos_y,
			width => $self->cell_size - $self->cell_gap * 2,
			height => $self->cell_size - $self->cell_gap * 2
		));
		$self->resources( $self->resources - $hero_cost );
	}
}

sub handle_heroes {
	my $self = shift;
	my $heroes = $self->heroes;
	for (my $i = 0; $i < scalar @{$heroes}; $i++) {
		$heroes->[$i]->draw($self);
		$heroes->[$i]->update($self);

		if ($self->alien_positions->indexOf($heroes->[$i]->y) != -1) {
			$heroes->[$i]->shooting(1);
		} else {
			$heroes->[$i]->shooting(0);
		}

		my $aliens = $self->aliens;
		for (my $j = 0; $j < scalar @{$aliens}; $j++) {
			if ($heroes->[$i] && $self->collision($heroes->[$i], $aliens->[$j])) {
				$aliens->[$j]->movement(0);
				$heroes->[$i]->health($heroes->[$i]->health - 0.2);
			}
		}

		if ($heroes->[$i]->health <= 0) {
			for (my $j = 0; $j < scalar @{$aliens}; $j++) {
				if ($self->collision($heroes->[$i], $aliens->[$j])) {
					$aliens->[$j]->movement($aliens->[$j]->speed);
				}
			}
			$heroes->splice($i, 1);
			$i--;
		}
	}
}

sub handle_aliens {
	my $self = shift;
	my $aliens = $self->aliens;
	for (my $i = 0; $i < scalar @{$aliens}; $i++) {
		$aliens->[$i]->draw($self);
		$aliens->[$i]->update($self);
		if ($aliens->[$i]->x < 0) {
			$self->game_over(1);
		}
		if ($aliens->[$i]->dead) {
			if ($aliens->[$i]->death_sprite->current_loop > 1) {
				$aliens->splice($i, 1);
				$i--;
			}
		} elsif ($aliens->[$i]->health <= 0) {
			my $gained = $aliens->[$i]->max_health / 10;
			$self->messages->push(Game::HeroesVsAliens::Message->new({
				value => '+' . $gained,
				x => 100,
				y => 10,
				size => 20,
				color => [207, 176, 21],
			}));
			$self->messages->push(Game::HeroesVsAliens::Message->new({
				value => '+' . $gained,
				x => $aliens->[$i]->x,
				y => $aliens->[$i]->y,
				size => 20,
				color => [0, 0, 0],
			}));
			$self->resources(
				$self->resources + $gained
			);
			$self->score(
				$self->score + $gained
			);
			my $index = $self->alien_positions->indexOf($aliens->[$i]->y);
			$self->alien_positions->splice($index, 1);
			$aliens->[$i]->dead(1);
		}
	}
	if ($self->frame % $self->aliens_interval == 0 && $self->score < $self->winning_score) {
		my $y_pos = int(rand(5) + 1) * $self->cell_size + $self->cell_gap;
		$self->aliens->push(Game::HeroesVsAliens::Alien->new({
			y => $y_pos,
			x => $self->app_width,
			width => $self->cell_size - $self->cell_gap * 2,
			height => $self->cell_size - $self->cell_gap * 2
		}));
		$self->alien_positions->push($y_pos);
		if ($self->aliens_interval > 10) {
			$self->aliens_interval($self->aliens_interval - int(rand(40)));
		}
	}
}

sub handle_messages {
	my ($self) = @_;
	my $messages = $self->messages;
	for (my $i = 0; $i < $messages->length; $i++) {
		$messages->[$i]->update($self);
		$messages->[$i]->draw($self);
		if ($messages->[$i]->life_span >= 50) {
			$messages->splice($i, 1);
			$i--;
		}
	}
}

sub handle_bonuses {
	my ($self) = @_;

	if ($self->frame % 500 == 0 && $self->score < $self->winning_score) {
		my $y_pos = int(rand(5) + 1) * $self->cell_size + $self->cell_gap;
		my $x_pos = int(rand(8)) * $self->cell_size + $self->cell_gap;
		$self->bonuses->push(Game::HeroesVsAliens::Bonus->new({
			x => $x_pos,
			y => $y_pos
		}));
	}

	my $bonuses = $self->bonuses;
	for (my $i = 0; $i < $bonuses->length; $i++) {
		$bonuses->[$i]->draw($self);
		if ($bonuses->[$i] && $self->collision($self->mouse, $bonuses->[$i])) {
			my $gained = $bonuses->[$i]->amount;
			
			$self->resources($self->resources + $gained);
			$self->messages->push(Game::HeroesVsAliens::Message->new({
				value => '+' . $gained,
				x => $bonuses->[$i]->x,
				y => $bonuses->[$i]->y,
				size => 20,
				color => [0, 0, 0]
			}));
			$self->messages->push(Game::HeroesVsAliens::Message->new({
				value => '+' . $gained,
				x => 130,
				y => 30,
				size => 20,
				color => [207, 176, 21]
			}));
			$bonuses->splice($i, 1);
			$i--;
		}
	}




}

sub collision {
	my ($self, $first, $second) = @_;
	if (
		!(
			$first->x > $second->x + $second->width ||
			$first->x + $first->width < $second->x ||
			$first->y > $second->y + $second->width ||
			$first->y + $first->height < $second->y
		)
	) {
		return 1;
	}
	return 0;
}

sub draw_rect {
	my ($self, $x, $y, $w, $h, $colour) = @_;
        my $rect = $self->ctx->draw_rect([$x, $y, $w, $h], $colour );
	$self->ctx->update;
}

sub draw_outline_rect {
	my ($self, $x, $y, $w, $h, $r, $g, $b, $a) = @_;
	SDL::GFX::Primitives::rectangle_RGBA(
		$self->ctx,
		$x,
		$y,
		$x + $w,
		$y + $h,
		$r,
		$g,
		$b,
		$a
	);
}

sub draw_text {
	my ($self, $x, $y, $size, $color, $text) = @_;
	my $draw = SDLx::Text->new(
		size => $size,
		x => $x,
		y => $y,
		color => $color
	);
	$draw->write_to($self->ctx, $text);
}

sub draw_arc {
	my ($self, $x, $y, $w, $h, $color) = @_;
	SDL::GFX::Primitives::arc_RGBA(
		$self->ctx,
		$x,
		$y,
		$w,
		-90,
		90,
		@{$color}
	);
}

sub draw_circle {
	my ($self, $x, $y, $w, $h, $color) = @_;
	SDL::GFX::Primitives::filled_circle_RGBA(
		$self->ctx,
		$x,
		$y,
		$w,
		@{$color}
	);
}

1;

__END__

=head1 NAME

Game::HeroesVsAliens - A tower defense game.

=head1 VERSION

Version 0.01

=cut

=head1 SYNOPSIS

	SDLPerl /bin/HeroesVsAliens.pl

=head1 Description

Game::HeroesVsAliens is a simple tower defense game using SDL. 

=head1 AUTHOR

lnation, C<< <thisusedtobeanemail at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-game-heroesvsaliens at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Game-HeroesVsAliens>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Game::HeroesVsAliens

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<https://rt.cpan.org/NoAuth/Bugs.html?Dist=Game-HeroesVsAliens>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/Game-HeroesVsAliens>

=item * Search CPAN

L<https://metacpan.org/release/Game-HeroesVsAliens>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2021 by lnation.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)


=cut
